"""Script for running tsunami inundation scenario for Dampier, WA, Australia.

Source data such as elevation and boundary data is assumed to be available in
directories specified by project.py
The output sww file is stored in project.output_run_time_dir 

The scenario is defined by a triangular mesh created from project.polygon,
the elevation data and a simulated tsunami generated with URS code.

Ole Nielsen and Duncan Gray, GA - 2005 and Jane Sexton, Nick Bartzis, GA - 2006
"""

#------------------------------------------------------------------------------
# Import necessary modules
#------------------------------------------------------------------------------

# Standard modules
from os import sep
from os.path import dirname, basename
from os import mkdir, access, F_OK
from shutil import copy
import time
import sys

# Related major packages
from anuga.shallow_water import Domain
from anuga.shallow_water import Dirichlet_boundary
from anuga.shallow_water import File_boundary
from anuga.shallow_water import Reflective_boundary
from anuga.shallow_water import Field_boundary
from Numeric import allclose

from anuga.pmesh.mesh_interface import create_mesh_from_regions
from anuga.abstract_2d_finite_volumes.util import start_screen_catcher, copy_code_files
from anuga_parallel.parallel_api import distribute, numprocs, myid, barrier
from anuga_parallel.parallel_abstraction import get_processor_name
from anuga.caching import myhash
# Application specific imports
import project                 # Definition of file names and polygons

#------------------------------------------------------------------------------
# Copy scripts to time stamped output directory and capture screen
# output to file
#------------------------------------------------------------------------------

#copy script must be before screen_catcher
if myid == 0:
    copy_code_files(project.output_run_time_dir,__file__, 
                 dirname(project.__file__)+sep+ project.__name__+'.py' )
barrier()

start_screen_catcher(project.output_run_time_dir, myid, numprocs)

print "Processor Name:",get_processor_name()
barrier()

# filenames
#boundaries_name = project.boundaries_name
meshes_dir_name = project.meshes_dir_name+'.msh'
#boundaries_dir_name = project.boundaries_dir_name

tide = project.tide

# creates copy of code in output dir


print 'USER: ', project.user
print 'min triangles', project.trigs_min,
print 'Note: This is generally about 20% less than the final amount'

#--------------------------------------------------------------------------
# Create the triangular mesh based on overall clipping polygon with a
# tagged
# boundary and interior regions defined in project.py along with
# resolutions (maximal area of per triangle) for each polygon
#--------------------------------------------------------------------------

if myid == 0:
    
    print 'start create mesh from regions'

    create_mesh_from_regions(project.poly_all,
                             boundary_tags={'back': [2,3], 'side': [0, 1, 4],
                                            'ocean': [5]},
                             maximum_triangle_area=project.res_poly_all,
                             interior_regions=project.interior_regions,
                             filename=meshes_dir_name,
                             use_cache=False,
                             verbose=True)

# to sync all processors are ready 
barrier()

#-------------------------------------------------------------------------
# Setup computational domain
#-------------------------------------------------------------------------
print 'Setup computational domain'
#from caching import cache

#domain = cache(Domain, (meshes_dir_name), {'use_cache':True, 'verbose':True}, verbose=True)
#above don't work
domain = Domain(meshes_dir_name, use_cache=False, verbose=True)
print 'domain id', id(domain) 
print 'myhash', myhash(domain)      
       
print domain.statistics()

boundaries_dir_name=project.boundaries_dir_name

print 'starting to create boundary conditions'

from anuga.shallow_water.data_manager import urs2sww


#-------------------------------------------------------------------------
# Setup initial conditions
#-------------------------------------------------------------------------
if myid == 0:

    print 'Setup initial conditions'

    from polygon import *
    #following sets the stage/water to be offcoast only
    IC = Polygon_function( [(project.poly_mainland, 0.)], default = tide)
#    IC = Polygon_function( [(project.poly_mainland, 0.)], default = 200)
    domain.set_quantity('stage', IC)
    domain.set_quantity('friction', 0.01) 
    print 'Start Set quantity'

    domain.set_quantity('elevation', 
#                    filename = project.combined_dir_name + '.pts',
# MUST USE TXT FILES FOR CACHING TO WORK!
                    filename = project.combined_dir_name + '.txt',
#                    filename = project.combined_smallest_dir_name + '.txt',
                    use_cache = True,
                    verbose = True,
                    alpha = 0.1)
    print 'Finished Set quantity'
barrier()


#------------------------------------------------------
# Distribute domain to implement parallelism !!!
#------------------------------------------------------

if numprocs > 1:
    domain=distribute(domain)

#------------------------------------------------------
# Set domain parameters
#------------------------------------------------------
print 'domain id', id(domain)
domain.set_name(project.scenario_name)
domain.set_datadir(project.output_run_time_dir)
domain.set_default_order(2) # Apply second order scheme
domain.set_minimum_storable_height(0.01) # Don't store anything less than 1cm
domain.set_store_vertices_uniquely(False)
domain.set_quantities_to_be_stored(['stage', 'xmomentum', 'ymomentum'])
domain.set_maximum_allowed_speed(0.1) # Allow a little runoff (0.1 is OK)
print 'domain id', id(domain)
domain.beta_h = 0
#domain.limit2007 = 1

#-------------------------------------------------------------------------
# Setup boundary conditions 
#-------------------------------------------------------------------------
print 'Available boundary tags', domain.get_boundary_tags()
print 'domain id', id(domain)
#print 'Reading Boundary file',project.boundaries_dir_namea + '.sww'

Bf = Field_boundary(project.boundaries_dir_namea + '.sww',
                    domain, time_thinning=1, mean_stage=tide, 
                    use_cache=True, verbose=True)

print 'finished reading boundary file'

Br = Reflective_boundary(domain)
Bd = Dirichlet_boundary([tide,0,0])

print'set_boundary'
##domain.set_boundary({'back': Br,
##                     'side': Bf,
##                     'ocean': Bf})
domain.set_boundary({'back': Br,
                     'side': Bd,
                     'ocean': Bf}) 
print'finish set boundary'

#----------------------------------------------------------------------------
# Evolve system through time
#----------------------------------------------------------------------------

t0 = time.time()

for t in domain.evolve(yieldstep = 60, finaltime = 29950):
    domain.write_time()
#    domain.write_time(track_speeds=True)
    domain.write_boundary_statistics(tags = 'ocean')
    
#    domain.write_time(track_speeds=True)

#for t in domain.evolve(yieldstep = 120, finaltime = 9000):
#    domain.write_time()
#    domain.write_boundary_statistics(tags = 'ocean')
'''     
for t in domain.evolve(yieldstep = 60, finaltime = 28800
                       ,skip_initial_step = True):
    domain.write_time()
    domain.write_boundary_statistics(tags = 'ocean')    

for t in domain.evolve(yieldstep = 120, finaltime = 34800
                       ,skip_initial_step = True): 
    domain.write_time()
    domain.write_boundary_statistics(tags = 'ocean')    
'''
x, y = domain.get_maximum_inundation_location()
q = domain.get_maximum_inundation_elevation()

print 'Maximum runup observed at (%.2f, %.2f) with elevation %.2f' %(x,y,q)

print 'That took %.2f seconds' %(time.time()-t0)

