#------------------------------------------------------------------------------
# Import necessary modules
#------------------------------------------------------------------------------

# Standard modules
import os
import time
import sys

# Related major packages
from anuga.shallow_water import Domain
from anuga.shallow_water import Transmissive_boundary
from anuga.shallow_water import Reflective_boundary
from anuga.shallow_water import Dirichlet_boundary
from anuga.shallow_water import Time_boundary
from anuga.shallow_water import File_boundary
from anuga.shallow_water import Field_boundary

from anuga.pmesh.mesh_interface import create_mesh_from_regions
from anuga.shallow_water.data_manager import convert_dem_from_ascii2netcdf
from anuga.shallow_water.data_manager import dem2pts
from anuga.shallow_water import Transmissive_Momentum_Set_Stage_boundary
from anuga.abstract_2d_finite_volumes.util import file_function

from anuga.caching import cache
from anuga.utilities.polygon import read_polygon, plot_polygons, \
                                    polygon_area, is_inside_polygon
from math import sin, pi, exp

				    
#------------------------------------------------------------------------------
# Define scenario as either ...
#------------------------------------------------------------------------------
scenario = 'test_32_vs_64'

if os.access(scenario, os.F_OK) == 0:
    os.mkdir(scenario)

basename = scenario + '_source'

#------------------
# Initial condition
#------------------
tide = 0.0


#------------------------------------------------------------------------------
# Preparation of topographic data
# Convert ASC 2 DEM 2 PTS using source data and store result in source data
#------------------------------------------------------------------------------

# Filenames
dem_name = 'bathy.txt' 
meshname = scenario+'.msh'


#---------
# Polygons
#---------

# bounding polygon for study area
bounding_polygon = read_polygon('extent.csv')

print 'Area of bounding polygon in km?', polygon_area(bounding_polygon)/1000000.0


#------------
# Resolutions
#------------
remainder_res = 5000 #100m

interior_regions = [[bounding_polygon, remainder_res]]

create_mesh_from_regions(bounding_polygon,
			     boundary_tags={'south': [0],
                                            'east': [1],
					    'north': [2],
					    'west': [3]},
			     maximum_triangle_area=remainder_res,
			     filename=meshname,
			     interior_regions=interior_regions,
			     use_cache=True, verbose=True)

#--------------------------------------------------------------------------
# Setup computational domain
#--------------------------------------------------------------------------

domain = Domain(meshname, use_cache=True, verbose=True)

print 'Number of triangles = ', len(domain)
print 'The extent is ', domain.get_extent()
print domain.statistics()

domain.set_name(basename) 
domain.set_datadir(scenario)
domain.set_quantities_to_be_stored(['stage', 'xmomentum', 'ymomentum'])
domain.set_minimum_storable_height(0.01)

#print 'domain.tight_slope_limiters', domain.tight_slope_limiters
#domain.tight_slope_limiters = 1
#print 'domain.tight_slope_limiters', domain.tight_slope_limiters

#--------------------------------------------------------------------------
# Setup initial conditions
#--------------------------------------------------------------------------

domain.set_quantity('stage', tide)
domain.set_quantity('friction', 0.025) 
domain.set_quantity('elevation', filename=dem_name,
			use_cache=True, verbose=True, alpha=0.01)



#------------------------------------------------------------------------------
# Setup boundary conditions
#------------------------------------------------------------------------------

print 'Available boundary tags', domain.get_boundary_tags()
Br = Reflective_boundary(domain)
Bd = Dirichlet_boundary([tide,0,0])
Bw = Transmissive_Momentum_Set_Stage_boundary(domain = domain,
                          function=lambda t: [(60<t<120)*5, 0, 0])

   
# Boundary condition for sww feed at the east boundary
domain.set_boundary({'south':Bd,
                     'east': Bw,
                     'north': Bd,
                     'west': Bd})
    
#------------------------------------------------------------------------------
# Evolve system through time
#------------------------------------------------------------------------------

import time
#t0 = time.time()
t0 = time.time()+6000

from Numeric import allclose
from anuga.abstract_2d_finite_volumes.quantity import Quantity


# nothing
for t in domain.evolve(yieldstep = 10, finaltime = 1800):
    domain.write_time()
    domain.write_boundary_statistics(tags = ['south','east','north'])

           
print 'That took %.2f seconds' %(time.time()-t0)
